package gov.va.genisis2.ts.service.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.jena.update.UpdateFactory;
import org.apache.jena.update.UpdateRequest;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import gov.va.genisis2.ts.common.dto.TripleDTO;
import gov.va.genisis2.ts.common.dto.UpdateResponseDTO;
import gov.va.genisis2.ts.common.dto.UpdateTripleDTO;
import gov.va.genisis2.ts.converter.UpdateResponseConverter;
import gov.va.genisis2.ts.fuseki.FusekiClient;
import gov.va.genisis2.ts.service.IUpdateTripleService;
import gov.va.genisis2.ts.utils.TSPropertiesUtil;

@Service
public class UpdateTripleService implements IUpdateTripleService {

	private static final Logger LOGGER = LogManager.getLogger(UpdateTripleService.class);
	@Autowired
	private FusekiClient fusekiClient;

	@Autowired
	private TSPropertiesUtil propsUtil;

	@Autowired
	private UpdateResponseConverter updateResponseConveter;

	@Autowired
	private MvpUriGenHelper mvpUriGenHelper;

	private static final String MVP_PREFIX = "http://URL";

	private static final String MVP_PREFIX_BRACKET = "<http://URL";

	private static final String INSERT = "INSERT DATA";

	private static final String DELETE = "DELETE DATA";

	public List<UpdateResponseDTO> deleteTriple(UpdateTripleDTO updateTripleDTO) {
		return updateTriple(DELETE, updateTripleDTO);
	}

	public List<UpdateResponseDTO> addTriple(UpdateTripleDTO updateTripleDTO) {
		return updateTriple(INSERT, updateTripleDTO);
	}

	public List<UpdateResponseDTO> updateTriple(String command, UpdateTripleDTO updateTripleDTO) {

		// removing all triples that have invalid prefixes in all cases

		// TODO: dropping triples that have no prefix...it may be best to handle this
		// filter
		// in the error processing logic for each case
		List<TripleDTO> mvpTriples = validateTriplesForPrefix(updateTripleDTO.getProperties());

		// populate URIs for new concepts
		mvpUriGenHelper.populateUrisForNewConcepts(mvpTriples);

		updateTripleDTO.setProperties(mvpTriples);

		List<UpdateResponse> queryResponses = createQueriesAndResponse(command, updateTripleDTO);

		if (updateTripleDTO.isAllowErrors()) {

			if (command.equalsIgnoreCase(INSERT)) {
				List<UpdateResponseDTO> dtoResponses = allowErrors(queryResponses).stream().map(dto -> updateResponseConveter.convert(dto)).collect(Collectors.toList());

				return dtoResponses;
			} else {
				List<UpdateResponseDTO> dtoResponses = allowErrorsDelete(queryResponses).stream().map(dto -> updateResponseConveter.convert(dto)).collect(Collectors.toList());
				return dtoResponses;
			}
		} else if (updateTripleDTO.isCheckOnly()) {
			// check and log for excetions as requests are processed..do not execute
			List<UpdateResponseDTO> dtoResponses = check(queryResponses).stream().map(dto -> updateResponseConveter.convert(dto)).collect(Collectors.toList());

			return dtoResponses;

		} else if (updateTripleDTO.isNoErrors()) {
			if (command.equalsIgnoreCase(INSERT)) {
				List<UpdateResponseDTO> dtoResponses = noErrors(queryResponses).stream().map(dto -> updateResponseConveter.convert(dto)).collect(Collectors.toList());

				return dtoResponses;
			} else {
				List<UpdateResponseDTO> dtoResponses = noErrorsDelete(queryResponses).stream().map(dto -> updateResponseConveter.convert(dto)).collect(Collectors.toList());

				return dtoResponses;
			}
		} else {
			// fall back to ignore errors case
			List<UpdateResponseDTO> dtoResponses = ignoreErrors(queryResponses).stream().map(dto -> updateResponseConveter.convert(dto)).collect(Collectors.toList());

			return dtoResponses;
		}
	}

	/**
	 * Checks for valid prefixes. If a triple does not have the correct mvp pattern
	 * it is removed
	 * 
	 * @param properties
	 * @return
	 */
	private List<TripleDTO> validateTriplesForPrefix(List<TripleDTO> properties) {

		List<TripleDTO> newTriples = properties.stream()
				.filter(t -> (t.getS().startsWith(MVP_PREFIX) || t.getS().startsWith(MVP_PREFIX_BRACKET)) || (t.getP().startsWith(MVP_PREFIX) || t.getP().startsWith(MVP_PREFIX_BRACKET)))
				.collect(Collectors.toList());
		return newTriples;
	}

	/**
	 * Creates mulitple query requests from dto information
	 * 
	 * @param command
	 * @param dto
	 * @return
	 */
	public UpdateRequest convertDTOtoRequests(String command, UpdateTripleDTO dto) {
		UpdateRequest request = UpdateFactory.create();

		for (TripleDTO prop : dto.getProperties()) {
			String query = command + " { " + quoteUri(prop.getS()) + " " + quoteUri(prop.getP()) + " " + quoteLiteral(prop.getO()) + ";}";
			request.add(query);
		}
		return request;
	}

	/**
	 * Format queries from DTO object into query form
	 * 
	 * @param command
	 * @param dto
	 * @return
	 */
	public List<String> createQueries(String command, UpdateTripleDTO dto) {
		List<String> queries = new ArrayList<String>();

		for (TripleDTO prop : dto.getProperties()) {
			queries.add(command + " { " + quoteUri(prop.getS()) + " " + quoteUri(prop.getP()) + " " + quoteLiteral(prop.getO()) + ";}");
		}
		return queries;
	}

	public List<UpdateResponse> check(List<UpdateResponse> queryResponse) {
		// check prefix
		UpdateRequest request = UpdateFactory.create();

		for (UpdateResponse qr : queryResponse) {
			try {
				request.add(qr.getQuery());
				// since this call goes over the network would only like to execute
				// if the query is syntatically correct
				if (fusekiClient.performAsk(qr.getAskForm(), propsUtil.getSparqlEndpoint())) {
					// duplicate query!
					qr.setDuplicate(true);
				}
			} catch (Exception e) {
				qr.setError(e.getMessage());
			}
		}
		return queryResponse;
	}

	public List<UpdateResponse> noErrors(List<UpdateResponse> queryResponse) {
		UpdateRequest request = UpdateFactory.create();

		boolean errors = false;
		for (UpdateResponse qr : queryResponse) {
			try {
				// checks to see if there is a duplicate note the endpoint used!
				if (fusekiClient.performAsk(qr.getAskForm(), propsUtil.getSparqlEndpoint())) {
					// duplicate query!
					qr.setDuplicate(true);
					errors = true;
					break; // no errors are allowed so break out
				}
				request.add(qr.getQuery());
			} catch (Exception e) {
				errors = true;
				qr.setError(e.getMessage());
			}
		}
		/// if there are no errors then execute them!
		if (errors) {
			return queryResponse;
		} else {
			try {
				fusekiClient.performUpdate(request, propsUtil.getSparqlEndpointUpdate());
			} catch (Exception e) {
				LOGGER.error("Error in execution update SPARQL query", e);
				throw e;
			}
			return queryResponse;
		}
	}

	public List<UpdateResponse> noErrorsDelete(List<UpdateResponse> queryResponse) {
		UpdateRequest request = UpdateFactory.create();

		boolean errors = false;
		for (UpdateResponse qr : queryResponse) {
			try {
				// checks to see if there is not a duplicate note the endpoint used!
				if (!fusekiClient.performAsk(qr.getAskForm(), propsUtil.getSparqlEndpoint())) {
					// duplicate query!
					qr.setDuplicate(true);
					errors = true;
					break; // no errors are allowed so break out
				}
				request.add(qr.getQuery());
			} catch (Exception e) {
				errors = true;
				qr.setError(e.getMessage());
			}
		}
		/// if there are no errors then execute them!
		if (errors) {
			return queryResponse;
		} else {
			try {
				fusekiClient.performUpdate(request, propsUtil.getSparqlEndpointUpdate());
			} catch (Exception e) {
				LOGGER.error("Error in execution update SPARQL query", e);
				throw e;
			}
			return queryResponse;
		}
	}

	public List<UpdateResponse> allowErrors(List<UpdateResponse> queryResponse) {
		UpdateRequest request = UpdateFactory.create();

		for (UpdateResponse qr : queryResponse) {
			try {
				// checks to see if there is a duplicate. Note the endpoint used!
				if (fusekiClient.performAsk(qr.getAskForm(), propsUtil.getSparqlEndpoint())) {
					// duplicate query!
					qr.setDuplicate(true);
					continue; // we do not want to add this request because it is a duplicate
				}
				request.add(qr.getQuery());
			} catch (Exception e) {
				qr.setError(e.getMessage());
			}
		}

		try {
			fusekiClient.performUpdate(request, propsUtil.getSparqlEndpointUpdate());
		} catch (Exception e) {
			LOGGER.error("Error in execution update SPARQL query", e);
			throw e;
		}
		return queryResponse;
	}

	public List<UpdateResponse> allowErrorsDelete(List<UpdateResponse> queryResponse) {
		UpdateRequest request = UpdateFactory.create();

		for (UpdateResponse qr : queryResponse) {
			try {
				// checks to see if there is not a duplicate note the endpoint used!
				if (!fusekiClient.performAsk(qr.getAskForm(), propsUtil.getSparqlEndpoint())) {
					// duplicate query!
					qr.setDuplicate(false);
					continue; // we do not want to add this request because it is a duplicate
				}
				request.add(qr.getQuery());
			} catch (Exception e) {
				qr.setError(e.getMessage() + "\n");
			}
		}
		try {
			fusekiClient.performUpdate(request, propsUtil.getSparqlEndpointUpdate());

		} catch (Exception e) {
			LOGGER.error("Error in execution update SPARQL query", e);
			throw e;
		}
		return queryResponse;
	}

	public List<UpdateResponse> ignoreErrors(List<UpdateResponse> queryResponse) {
		UpdateRequest request = UpdateFactory.create();

		for (UpdateResponse qr : queryResponse)
			request.add(qr.getQuery());

		try {
			fusekiClient.performUpdate(request, propsUtil.getSparqlEndpointUpdate());

		} catch (Exception e) {
			LOGGER.error("Error in execution update SPARQL query", e);
			throw e;
		}
		return queryResponse;
	}

	/**
	 * Create sparql formed queries for the given triples
	 * 
	 * @param command
	 * @param dto
	 * @return
	 */
	public List<UpdateResponse> createQueriesAndResponse(String command, UpdateTripleDTO dto) {
		List<UpdateResponse> queries = new ArrayList<UpdateResponse>();

		for (TripleDTO prop : dto.getProperties()) {
			String askForm = "ASK { " + quoteUri(prop.getS()) + " " + quoteUri(prop.getP()) + " " + quoteLiteral(prop.getO()) + ";}";
			String query = command + " { " + quoteUri(prop.getS()) + " " + quoteUri(prop.getP()) + " " + quoteLiteral(prop.getO()) + ";}";

			queries.add(new UpdateResponse(query, "", askForm));
		}
		return queries;
	}

	/**
	 * Adds angular brackets to resources.. This is done because SPARQL needs
	 * resource to be in angular brackets
	 * 
	 * @param uri
	 * @return
	 */
	public String quoteUri(String uri) {
		if (uri.startsWith("<")) {
			return uri;
		}

		return "<" + uri + ">";
	}

	/**
	 * Adds double quotes to literals.. This is done because SPARQL needs literals
	 * to be double quoted
	 * 
	 * @param query
	 * @return
	 */
	public String quoteLiteral(String query) {
		if (query.startsWith("<")) {
			return query;
		}
		return "\"" + query + "\"";
	}
}
